<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set JSON header
header('Content-Type: application/json');

// Check if user has permission to view customers
if (!hasPermission('view_customers')) {
    http_response_code(403);
    echo json_encode(['error' => 'Permission denied']);
    exit;
}

if (!isset($_GET['customer_id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Customer ID is required']);
    exit;
}

$customerId = intval($_GET['customer_id']);

try {
    // First verify the customer exists
    $checkStmt = $pdo->prepare("SELECT id FROM customers WHERE id = ?");
    $checkStmt->execute([$customerId]);
    if (!$checkStmt->fetch()) {
        http_response_code(404);
        echo json_encode(['error' => 'Customer not found']);
        exit;
    }

    // Get all transactions ordered by date
    $query = "SELECT created_at, type, amount 
              FROM credit_transactions 
              WHERE customer_id = ? 
              ORDER BY created_at ASC";
              
    $stmt = $pdo->prepare($query);
    $stmt->execute([$customerId]);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate running balance
    $balance = 0;
    $formattedTransactions = [];
    
    foreach ($transactions as $transaction) {
        if ($transaction['type'] === 'credit') {
            $balance += floatval($transaction['amount']);
        } else {
            $balance -= floatval($transaction['amount']);
        }
        
        $formattedTransactions[] = [
            'created_at' => $transaction['created_at'],
            'type' => $transaction['type'],
            'amount' => floatval($transaction['amount']),
            'description' => $transaction['type'] === 'credit' ? 'Credit Purchase' : 'Payment Made',
            'balance' => $balance
        ];
    }
    
    // Reverse the array to show newest first
    $formattedTransactions = array_reverse($formattedTransactions);
    
    echo json_encode($formattedTransactions);
    
} catch (PDOException $e) {
    error_log("Credit history database error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database error occurred']);
} catch (Exception $e) {
    error_log("Credit history general error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'An unexpected error occurred']);
} 